#include <stdio.h>
#include <sys/io.h>
#include <pthread.h>
#include "sio_acce.h"

#define MUTEX

#ifdef MUTEX
static pthread_mutex_t accessing = PTHREAD_MUTEX_INITIALIZER;
#endif
 

/* Read one byte from SIO register */
static unsigned char readSioByte(unsigned short reg);

/* Write one byte to SIO register */
static void writeSioByte(unsigned short reg, unsigned char data);

/* Read one byte frome a register in SIO Logic Device */
static unsigned char readLogDevByte(unsigned char dev_num, unsigned short reg);

/* Write one byte to a register in SIO Logic Device */
static void writeLogDevByte(unsigned char dev_num, unsigned short reg, unsigned char data);

/* Configure GPIO30 ~ GPIO37 to the expected operating status */
static void setupGpio(void);

/* Reset Watchdog timer */
static void resetWtd(void);

/* Configure Watchdog to the expected operating status */
static void setupWtd(void);

/* Configure pin GP32, GP33, GP34 as GPIO */
void setupGpio(void)
{
  unsigned char retval = 0x0;

  /* Configure GP32, GP33, GP34 as GPIO */
  retval = readSioByte( SIO_MFUNC_PIN_REG1 );
  retval = retval & ~0xe0; // mask out bit 5~7
  writeSioByte( SIO_MFUNC_PIN_REG1, retval);
  retval = 0x0;

  /* Disable eventroute of GP30, GP31, GP35 to PSOUT# and PME# */
  retval = readLogDevByte( ACPI_DEV, 0xfe );  
  retval = retval & SIO_GPIO_EVTROUT;
  writeLogDevByte( ACPI_DEV, 0xfe, retval );
  retval = 0x0;

  /* Configure pin GP30 ~ GP37 as GPIO, not SUSLED */
  writeLogDevByte( GPIO_DEV, SIO_GPIO_MFUNC, 0x0);

  /* Make sure the signals at GP30 ~ GP37 are not inverted. */
  writeLogDevByte( GPIO_DEV, SIO_GPIO_IVRSN, 0x0 );

  /* Activate GPIO3 */
  retval = readLogDevByte( GPIO_DEV, SIO_GPIO_ACT_REG ); 
  retval = retval | SIO_GPIO_ACT;
  writeLogDevByte( GPIO_DEV, SIO_GPIO_ACT_REG, retval );
  retval = 0x0;

  /* Configure GP30 ~ GP33 as Output, GP34 ~ GP37 as Input */
  writeLogDevByte( GPIO_DEV, SIO_GPIO_CONF, SIO_GPIO_SET_IO );

  return;
}

unsigned char readSioByte(unsigned short reg)
{
  unsigned char retval = 0x0;

	iopl(3);

  /* Enter the Extneded Function Mode */
  outb( 0x87, SIO_EFER );
  outb( 0x87, SIO_EFER );

  /* Access Registers */
  outb( reg, SIO_EFIR );
  retval = inb( SIO_EFDR );

  /* Exit the Extneded Function Mode */
  outb( 0xAA, SIO_EFER);

	iopl(0);

  return( retval );
}


void writeSioByte(unsigned short reg, unsigned char data)
{
	iopl(3);

  /* Enter the Extneded Function Mode */
  outb( 0x87, SIO_EFER );
  outb( 0x87, SIO_EFER );

  /* Access Registers */
  outb( reg,  SIO_EFIR );

  outb( data, SIO_EFDR );

  /* Exit the Extneded Function Mode */
  outb( 0xAA, SIO_EFER);
 
	iopl(0);

  return;
}


unsigned char readLogDevByte(unsigned char dev_num, unsigned short reg)
{
  unsigned char retval = 0x0;

	iopl(3);

  /* Enter the Extneded Function Mode */
  outb( 0x87, SIO_EFER );
  outb( 0x87, SIO_EFER );

  /* Select device */
  outb( SIO_SEL_DEV, SIO_EFIR );
  outb( dev_num, SIO_EFDR );

  /* Access Registers */
  outb( reg, SIO_EFIR );

  retval = inb( SIO_EFDR );

  /* Exit the Extneded Function Mode */
  outb( 0xAA, SIO_EFER);

	iopl(0);

  return( retval );
}


void writeLogDevByte(unsigned char dev_num, unsigned short reg, unsigned char data)
{
	iopl(3);

  /* Enter the Extneded Function Mode */
  outb( 0x87, SIO_EFER );
  outb( 0x87, SIO_EFER );

  /* Select device */
  outb( SIO_SEL_DEV, SIO_EFIR );
  outb( dev_num, SIO_EFDR );

  /* Access Registers */
  outb( reg,  SIO_EFIR );

  outb( data, SIO_EFDR );

  /* Exit the Extneded Function Mode */
  outb( 0xAA, SIO_EFER);
 
	iopl(0);

  return;
}

/* Restore WDTO# if it has ever been triggered */
void resetWtd(void)
{
  unsigned char retval = 0x0;

  retval = readLogDevByte( WTD_DEV, SIO_WTD_STATUS );
  retval = retval & SIO_WTD_RESTO;
  writeLogDevByte( WTD_DEV, SIO_WTD_STATUS, retval );

  return;
}

void setupWtd(void)
{
  unsigned char retval = 0x0;

  /* Configure pin 77 as WDTO# */
  retval = readSioByte( SIO_MFUNC_PIN_REG2 );
  retval = retval & SIO_WTD_PINEAB;
  writeSioByte( SIO_MFUNC_PIN_REG2, retval );
  retval = 0x0;

  /* Set WDTO# active */
  retval = readLogDevByte( WTD_DEV, SIO_WTD_ACT_REG);
  retval = retval | SIO_WTD_ACTIVE;
  writeLogDevByte( WTD_DEV, SIO_WTD_ACT_REG, retval);
  retval = 0x0;

  /* Set up timer unit to second */
  retval = readLogDevByte( WTD_DEV, SIO_WTD_CONTROL);
  retval = retval & SIO_WTD_SECOND;
  writeLogDevByte( WTD_DEV, SIO_WTD_CONTROL, retval);
  retval = 0x0;

  return;
}


/* -------------------------- API -------------------------- */

/*
 * Watchdog API
 */

unsigned char getWtdTimer(void) 
{
  static int wtdSetup = 0;
  unsigned char retval = 0x0;

#ifdef MUTEX
  pthread_mutex_lock( &accessing );
#endif

  if( !wtdSetup ) {
    setupWtd();
    wtdSetup = 1;
  }
    
  retval = readLogDevByte( WTD_DEV, SIO_WTD_COUNT );

#ifdef MUTEX
  pthread_mutex_unlock( &accessing );
#endif

  return(retval);
}


void setWtdTimer(unsigned char value)
{
  static int wtdSetup = 0;

#ifdef MUTEX
  pthread_mutex_lock( &accessing );
#endif

  if( !wtdSetup ) {
    setupWtd();
    wtdSetup = 1;
  }

  resetWtd();
  writeLogDevByte( WTD_DEV, SIO_WTD_COUNT, value);

#ifdef MUTEX
  pthread_mutex_unlock( &accessing );
#endif

  return;
}

/*
 * GPIO API
 */
int getInChLevel( int channel, unsigned char *val )
{

  static int GpioSetup = 0;
  unsigned char retval = 0;
  unsigned char mask   = 0;

#ifdef MUTEX
  pthread_mutex_lock( &accessing );
#endif

  if( channel < 0x0 || channel > 0xf ) {
#ifdef MUTEX
    pthread_mutex_unlock( &accessing );
#endif
    return(-1);
  }

  /* Check if GPIO has been configured properly. */
  if(!GpioSetup) {
    setupGpio();
    GpioSetup = 1;
  }


  /* The Input bits are bit 4 ~ bit 7 */
  mask = (unsigned char) channel << 4;

  retval = readLogDevByte( GPIO_DEV, SIO_GPIO_DATA );

  retval = retval & mask;
  retval = retval >> 4;

  *val = retval;

#ifdef MUTEX
  pthread_mutex_unlock( &accessing );
#endif

  return(0);
};

int setOutChLevel( int channel, unsigned char val )
{
  static int GpioSetup = 0;
  unsigned char retval = 0;
  unsigned char mask = 0;

#ifdef MUTEX
  pthread_mutex_lock( &accessing );
#endif

  /* The Output bits are bit 0 ~ bit 3 */
  if( channel < 0 || channel > 0xf ) {
#ifdef MUTEX
    pthread_mutex_unlock( &accessing );
#endif
    return(-1);
  }

  /* Bits can be set to 0 or 1 only.
     Other value is wrong. */
  if( (val != 0) && (val != 1) ) {
#ifdef MUTEX
    pthread_mutex_unlock( &accessing );
#endif
    return(-1);
  }
  
  /* Check if GPIO has been configured properly. */
  if(!GpioSetup) {
    setupGpio();
    GpioSetup = 1;
  }

  mask = (unsigned char) channel & 0xff;
  retval = readLogDevByte( GPIO_DEV, SIO_GPIO_DATA );

  if( val== 0 ) 
    retval = retval & ~mask;  // Set indicated bits to 0
  else
    retval = retval | mask;   // Set indicated bits to 1

  writeLogDevByte( GPIO_DEV, SIO_GPIO_DATA, retval );

#ifdef MUTEX
  pthread_mutex_unlock( &accessing );
#endif

  return(0);

}

int getOutChLevel( int channel, unsigned char *val )
{
  static int GpioSetup = 0;
  unsigned char retval = 0;
  unsigned char mask   = 0;

#ifdef MUTEX
  pthread_mutex_lock( &accessing );
#endif
  if( channel < 0x0 || channel > 0xf ) {
#ifdef MUTEX
    pthread_mutex_unlock( &accessing );
#endif
    return(-1);
  }

  /* Check if GPIO has been configured properly. */
  if(!GpioSetup) {
    setupGpio();
    GpioSetup = 1;
  }

  /* The Output bits are bit 0 ~ bit 3 */
  mask = (unsigned char) channel;

  retval = readLogDevByte( GPIO_DEV, SIO_GPIO_DATA );

  retval = retval & mask;

  *val = retval;

#ifdef MUTEX
  pthread_mutex_unlock( &accessing );
#endif
  return(0);
}
